import { NextRequest, NextResponse } from "next/server";
import { createAdminClient } from "@/utils/supabase/admin";
import { createClient } from "@/utils/supabase/server";

// 获取单个工具详情
export async function GET(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取工具信息
    const { data: tool, error } = await adminClient
      .from("ai_tools")
      .select(`
        *,
        category:categories(id, name)
      `)
      .eq("id", params.id)
      .single();
      
    if (error) {
      console.error("获取工具详情错误:", error);
      return NextResponse.json(
        { error: "获取工具详情失败" },
        { status: 500 }
      );
    }
    
    if (!tool) {
      return NextResponse.json(
        { error: "未找到指定工具" },
        { status: 404 }
      );
    }
    
    return NextResponse.json({ tool });
    
  } catch (error) {
    console.error("处理获取工具详情请求错误:", error);
    return NextResponse.json(
      { error: "处理获取工具详情请求失败" },
      { status: 500 }
    );
  }
}

// 更新工具信息
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取请求体
    const updateData = await request.json();
    
    // 验证必填字段
    if (!updateData.name || !updateData.slug || !updateData.description || 
        !updateData.website_url || !updateData.category_id) {
      return NextResponse.json(
        { error: "缺少必填字段" },
        { status: 400 }
      );
    }
    
    // 验证slug格式
    const slugRegex = /^[a-z0-9-]+$/;
    if (!slugRegex.test(updateData.slug)) {
      return NextResponse.json(
        { error: "URL短识只能包含小写字母、数字和连字符" },
        { status: 400 }
      );
    }
    
    // 验证slug唯一性 (排除当前工具本身)
    const { data: existingTool } = await adminClient
      .from("ai_tools")
      .select("id")
      .eq("slug", updateData.slug)
      .neq("id", params.id)
      .single();
      
    if (existingTool) {
      return NextResponse.json(
        { error: "该URL短识已被使用，请选择其他标识" },
        { status: 400 }
      );
    }
    
    // 添加更新时间
    const now = new Date().toISOString();
    const dataWithTimestamp = {
      ...updateData,
      updated_at: now
    };
    
    // 更新数据
    const { data, error } = await adminClient
      .from("ai_tools")
      .update(dataWithTimestamp)
      .eq("id", params.id)
      .select()
      .single();
      
    if (error) {
      console.error("更新工具错误:", error);
      return NextResponse.json(
        { error: "更新工具失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({ 
      message: "工具更新成功", 
      tool: data 
    });
    
  } catch (error) {
    console.error("处理更新工具请求错误:", error);
    return NextResponse.json(
      { error: "处理更新工具请求失败" },
      { status: 500 }
    );
  }
}

// 删除工具
export async function DELETE(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 删除数据
    const { error } = await adminClient
      .from("ai_tools")
      .delete()
      .eq("id", params.id);
      
    if (error) {
      console.error("删除工具错误:", error);
      return NextResponse.json(
        { error: "删除工具失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({ 
      message: "工具删除成功"
    });
    
  } catch (error) {
    console.error("处理删除工具请求错误:", error);
    return NextResponse.json(
      { error: "处理删除工具请求失败" },
      { status: 500 }
    );
  }
} 